<?php

namespace App\Http\Controllers\Accounting;

use App\BranchModel;
use App\Model\Accounting\AccountChartModel;
use App\Model\Accounting\AccountTypeModel;
use App\Model\Accounting\JournalModel;
use App\Model\Accounting\ProjectModel;
use App\Model\Accounting\VoucherModel;
use App\Model\SupplierModel;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Yajra\DataTables\DataTables;
use DB;
use Auth;
use Gate;

class JournalController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:Journal List', ['only' => ['index']]);
        $this->middleware('permission:Journal Create', ['only' => ['create','store']]);
        $this->middleware('permission:Journal Edit', ['only' => ['edit','update']]);
        $this->middleware('permission:Journal Delete', ['only' => 'confirm_delete']);
        $this->middleware('permission:Journal', ['only' => ['print_voucher']]);
    }

    public function index()
    {
        $acc_code = AccountChartModel::select('id', 'acc_name')
            ->pluck('acc_name', 'id')
            ->prepend('Please Select', '');

        $supplier = SupplierModel::pluck('name', 'id')
            ->prepend('Please Select', '');
        $branch = BranchModel::pluck('name', 'id')
            ->prepend('Please Select', '');
        return view('Accounting.journal.index', ['acc_code' => $acc_code, 'supplier' => $supplier, 'branch' => $branch]);
    }

    public function get_journal(Request $request)
    {
        if ($request->ajax()) {
            $from = (empty($request->start_date)) ? '' : date('Y-m-d', strtotime($request->start_date));
            $to = (empty($request->end_date)) ? '' : date('Y-m-d', strtotime($request->end_date));
            $ref = $request->reference_no;
            $data = DB::table('nso007_journal as jn')
                ->join('nso007_account_chart as chart', 'chart.id', '=', 'jn.journal_acccode')
                ->leftjoin('nso007_suppliers as supplier', 'supplier.id', 'jn.journal_by_supplier')
                ->leftjoin('nso007_branch', 'nso007_branch.id', '=', 'jn.branch_id')
                ->leftjoin('users', 'users.id', 'jn.journal_by')
                ->selectRaw(
                    'chart.acc_name,
                    chart.id as account_code,
                    jn.id,
                    jn.journal_tran_id,
                    jn.journal_des,
                    jn.journal_debit,
                    jn.journal_credit,
                    jn.journal_invoice,
                    jn.journal_paydate,
                    jn.journal_debit,
                    jn.journal_reference_number,
                    nso007_branch.name as branch_name,
                    supplier.name as supplier_name,
                    jn.voucher_type,
                    jn.voucher_ref,
                    jn.journal_audit_status,
                    jn.journal_tran_id as tran_id,
                    jn.journal_status,
                    jn.journal_currency,
                    jn.journal_referenceid,
                    jn.transaction_type,
                    users.name,SUM(jn.journal_debit) as debit,SUM(jn.journal_credit) as credit'
                )
                ->when(request('start_date', ''), function ($q) use ($from, $to) {
                    $q->whereBetween('journal_paydate', [$from, $to]);
                })
                ->where(function ($query) {
                    $query->where('jn.branch_id', FindBranchID());
                })
                ->where(function ($query) use ($ref) {
                    if ($ref) {
                        $query->where('jn.journal_reference_number', $ref);
                    }
                })
                ->orderby('jn.journal_tran_id', 'DESC')
                ->orderby('jn.journal_debit', 'DESC')
                ->orderby('jn.journal_credit', 'DESC')
                ->groupBy('jn.journal_tran_id')
                ->get();
            return DataTables::of($data)
                ->editColumn('journal_paydate', function ($data) {
                    return date('d-m-Y',strtotime($data->journal_paydate));
                })
                ->editColumn('voucher', function ($data) {
                    if ($data->voucher_type) {
                        return getVoucherType($data->voucher_type);
                    }
                })
                ->addColumn('action', function ($data) {
                    $btn_edit = '';
                    $btn_delete = '';
                    if ($data->journal_status == 0 && $data->journal_audit_status == 0) {
                        if (!$data->journal_referenceid) {
                            $btn_edit .= '<a data-href="' . route('journal.edit', $data->journal_tran_id) . '" data-text="Edit Journal" class="btn btn-warning form-control btn_action"><i class="fa fa-edit"></i></a> ';
                            $btn_delete .= ' <a data-href="' . route('journal_confirm_delete',$data->journal_tran_id) . '"  class="btn btn-danger btn_delete"><i class="fa fa-trash"></i></a> ';
                        }
                    }
                    return $btn_edit . $btn_delete;
                })
                ->editColumn('checkbox', function ($data) {
                    return '<label><input id="' . $data->journal_tran_id . '" class="checkbox" type="checkbox"></label>';
                })
                ->editColumn('debit', function ($data) {
                    return '<span data-curr="2" data-debit="' . ($data->debit) . '">$'. number_format($data->debit, 2) . '</span>';
                })
                ->editColumn('credit', function ($data) {
                    return '<span  data-curr="2" data-credit="' . ($data->credit) . '">$' . number_format($data->credit, 2) . '</span>';
                })
                ->editColumn('transaction_type', function ($data) {
                    if($data->transaction_type)
                    return __('administrator.'.$data->transaction_type);
                })
                ->rawColumns(['checkbox', 'tran_id','action','debit','credit'])
                ->make(true);
        }
    }

    public function print_voucher(Request $request)
    {
        $journal_tran_id = $request->journal_tran_id;
        $journal_tran_id = explode(",", $journal_tran_id);

        $journal = DB::table('nso007_journal as jn')
            ->join('nso007_account_chart as chart', 'chart.id', 'jn.journal_acccode')
            ->leftjoin('nso007_suppliers as supplier', 'supplier.id', '=', 'jn.journal_by_supplier')
            ->selectRaw('chart.acc_name,supplier.name,chart.acc_namekh,jn.*')
            ->whereIn('jn.journal_tran_id', $journal_tran_id)
            ->orderBy('jn.journal_debit','desc' )
            ->where('jn.branch_id',myBranch()->id)
            ->get();
        return view('Accounting.journal.report-print-voucher', ['data' => $journal]);
    }

    public function create()
    {
        $max = JournalModel::max('journal_tran_id');
        $voucher_ref = $max + 1;
        (string)$voucher_ref;
        while (strlen($voucher_ref) < 8) {
            $voucher_ref = "0" . $voucher_ref;
        }
        $voucher_ref = "V-" . $voucher_ref;
        return view('Accounting.journal.create', ['voucher_ref' => $voucher_ref]);
    }

    public function query_account_code(Request $request)
    {
        $account_code = $request->obj;
        $account_chart = AccountChartModel::where(DB::raw('substr(acctype,1,1)'), '=', $account_code)->get();
        echo \json_encode($account_chart);
    }

    public function store(Request $request)
    {
        return $this->update($request, null);
    }

    public function edit($id)
    {
        $journal = JournalModel::where('journal_tran_id',$id)->orderBy('id', 'asc')->get();
        return view('Accounting.journal.edit', compact('journal'));
    }

    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'date' => 'required',
            'voucher_ref' => 'required',
            'account_code.*' => 'required|exists:nso007_account_chart,id',
            'debit' => 'required',
            'credit' => 'required'
        ]);
        $max_id = JournalModel::max('id');
        $max_tran_journal = JournalModel::max('journal_tran_id');
        if (!$max_tran_journal) {
            $max_tran_journal = 0;
        }
        $journal_tran_id = $id??($max_tran_journal + 1);
        $date = date('Y-m-d', strtotime($request->date));
        if (!$max_id) {
            $max_id = 0;
        }
        $parent_id = ($max_id + 1);
        $account_code = $request->account_code;
        $this->journal_confirm_delete($id);
        foreach ($account_code as $key => $code) {
            $accType = AccountTypeModel::where('id', 'like', $code . '%')->first();
            $jType = $accType->acctype_group ?? $code;
            JournalModel::insert([
                'branch_id' => Auth::user()->branch_id,
                'journal_parentid' => $key>0?$parent_id:'',
                'journal_type' => $jType,
                'journal_acccode' => $code,
                'voucher_type' => $request->voucher_type,
                'voucher_ref' => $request->voucher_ref,
                'journal_des' => $request->note[$key],
                'journal_debit' => $request->debit[$key],
                'journal_credit' => $request->credit[$key],
                'journal_paydate' => $date,
                'journal_by' => Auth::user()->id,
                'journal_transactiondate' => date('Y-m-d', strtotime($request->date)),
                'journal_tran_id' => $journal_tran_id,
                'journal_by_project' => $request->journal_by_project,
                'journal_by_supplier' => $request->journal_by_supplier[$key],
                'journal_reference_number' => $request->journal_reference_number,
                'journal_currency' => 2,
                'journal_status' => 0
            ]);
        }

        return response()->json(['status' => 'ok','reload'=>1]);
    }

    public function journal_confirm_delete($id)
    {
        if($id){
            JournalModel::where('journal_tran_id', $id)->delete();
        }
        return response()->json(['status' => 'ok']);
    }
}
